#!/usr/bin/python3
# -*- coding: utf-8 -*-

from flask import request, jsonify, render_template, send_from_directory
import json
import hashlib
from datetime import datetime
from functools import wraps
from app import db

from app.models import *

from app import app
from app import jwt

from config import Config

from sqlalchemy import desc
from sqlalchemy.sql import text

from flask_jwt_extended import (
    JWTManager, jwt_required, create_access_token,
    jwt_refresh_token_required, create_refresh_token,
	decode_token, get_jwt_identity,	get_raw_jwt, get_jti,
	get_jwt_claims
)

from app.route_models import Models

def not_found_message(resource_type, resource_id):
    error = '{0} with id={1} was not found'.format(resource_type, resource_id)
    return jsonify({'errors': [error]})

def get_devid():
	return hashlib.md5((request.headers.get('sessionid') + request.headers.get('User-Agent') + Config.SECRET_KEY).encode()).hexdigest()

@jwt.user_claims_loader
def add_claims_to_access_token(identity):

	user = Users.query.filter(text('login="{}"'.format(identity))).first()

	groups = Groups.query.all()
	user_groups = []
	for group in groups:
		group_users = group.users.split(',')
		if identity in group_users:
			user_groups.append(group.name)

	return {
		'login': identity,
		'name': user.name,
		'groups': user_groups
	}

def JWTTockensGenerate(login):
	# Generate new tokens
	access_token = create_access_token(identity=login)
	refresh_token = create_refresh_token(identity=login)
	tokens = {
		'access_token': access_token,
		'refresh_token': refresh_token
	}

	###############################
	# Add refresh_token to database
	###############################
	devid = get_devid()
	jti = get_jti(refresh_token)
	exp = decode_token(refresh_token)['exp']
	try:
		new_data = Token({'login': login, 'devid': devid, 'jti': jti, 'refresh_token': refresh_token, 'exp': exp})
	except Exception:
		return jsonify({'errors': ['Bad data']}), 400

	try:
		db.session.add(new_data)
		db.session.commit()
	except Exception:
		db.session().rollback()
	###############################
	return tokens


##############################
# Access
##############################
from app.access import Access
access = Access()

def access_check(f):
	@wraps(f)
	def decorated_function(*args, **kwargs):
		claims = get_jwt_claims()
		if access.check(request, kwargs['model'], claims['login'], claims['groups']) == True:
			return f(*args, **kwargs)
		else:
			return jsonify({'errors':['Access deny']}), 403

	return decorated_function

# Routes:
##############################
# Static files
############################## 
@app.route('/frontend/')
def send_frontend_index():
	return send_from_directory('frontend/', 'index.html')

@app.route('/frontend/<path:path>')
def send_frontend_files(path):
	return send_from_directory('frontend/', path)

##############################
# Main Page
##############################
@app.route('/', methods=['GET'])
def home_page():
	return render_template('index.html'), 200

##############################
# LOGIN
##############################
def encodePassword(password):
	return hashlib.md5(password.encode()).hexdigest()

@app.route('/auth/login/', methods=['POST'])
def login():
	login = request.json.get('login')
	passwordHash = encodePassword(request.json.get('password'))
	query = Users.query.filter( text('login="{}" AND password="{}"'.format(login, passwordHash) ) )
	user = query.first()
	if user != None:
		# Generate new tokens
		tokens = JWTTockensGenerate(login)
		return jsonify(tokens), 200
	else:
		return jsonify({'errors': ['Bad username or password']}), 403

##############################
# REFRESH
##############################
@app.route('/auth/refresh/', methods=['POST'])
@jwt_refresh_token_required
def refresh():
	devid = get_devid()
	login = get_jwt_identity()

	# Delete old tokens from database
	timestamp = int(datetime.timestamp(datetime.now()))
	db.session.query(Token).filter(Token.exp< timestamp).delete()
	db.session.commit()

	# Find token into database
	jti = get_raw_jwt()['jti']
	token = Token.query.filter(text('login="{}" AND devid="{}" AND jti="{}"'.format(login, devid, jti))).first()

	if token:
		# Delete refresh token from database
		db.session.delete(token)
		db.session.commit()

		tokens = JWTTockensGenerate(login)
		return jsonify(tokens), 200
	else:
		return jsonify({'errors': ['Refresh token required']}), 403

##############################
# LOGOUT
##############################
@app.route('/auth/logout/', methods=['GET'])
@jwt_required
def logout():
	login = get_jwt_identity()
	devid = get_devid()
	# Delete tokens by 'login' & 'devid'
	db.session.query(Token).filter(Token.login==login).filter(Token.devid==devid).delete()
	db.session.commit()
	return jsonify({'msg': ['OK']}), 403

@app.route('/auth/logout-all/', methods=['GET'])
@jwt_required
def logout_all():
	login = get_jwt_identity()
	# Delete tokens by 'login'
	db.session.query(Token).filter(Token.login==login).delete()
	db.session.commit()
	return jsonify({'msg': ['OK']}), 403

##############################
# Create
##############################
@app.route('/<string:model>/', methods=['POST'])
@jwt_required
@access_check
def create_data(model):
	# Trying to create User object. Handle bad format exceptions(See models)
	try:
		new_data = Models[model]['class'](request.json)
	except Exception:
		return jsonify({'errors': ['Bad data']}), 400

	try:
		db.session.add(new_data)
		db.session.commit()
		return  Models[model]['schema'].jsonify(new_data), 200
	except Exception:
		db.session().rollback()
		return jsonify({'errors': ['Can not create']}), 409

##############################
# Get All Data from any Model
##############################
@app.route('/<string:model>/', methods=['GET'])
@jwt_required
@access_check
def get_data_all(model):
	data = Models[model]['class'].query.all()
	#if not data:
	#	return not_found_message(model, 'all'), 404
	if ('schemas' in Models[model]):
		return Models[model]['schemas'].jsonify(data), 200
	else:
		return jsonify(data), 200

##############################
# Get All Data from any Model with "while, order, pagination"
##############################
@app.route('/<string:model>/<string:prm>', methods=['GET'])
@jwt_required
@access_check
def get_data_prm(model, prm):
	per_page = 10
	page = 1

	query = Models[model]['class'].query

	try:
		prm = json.loads(prm)
	except Exception:
		prm = None

	if prm != None:
		if 'paginator' in prm:
			if 'limit' in prm['paginator'] and 'page' in prm['paginator']:
				per_page = int(prm['paginator']['limit'])
				page = int(prm['paginator']['page'])

		if 'search' in prm:
			filter_str = []
			for search in prm['search']:
				if search['operator'] == 'LIKE':
					search['field'] = 'CAST({0} as TEXT)'.format(search['field'])
					search['value'] = '"%{0}%"'.format(search['value'])
				filter_str.append( '{0} {1} {2}'.format(search['field'], search['operator'], search['value']) )
			
			filter = text(' OR '.join(filter_str))
			query = query.filter(filter)

		if 'order' in prm:
			order_param = prm['order'][0].split(' ')
			order_by = order_param[0]
			if len(order_param) > 1:
				if order_param[1] == 'desc':
					order_by = desc(order_param[0])
			query = query.order_by(order_by)

	data = query.paginate(page = page, per_page = per_page, error_out = False)
	items = Models[model]['schemas'].dump(data.items)
	record_count = {'_total_records_': data.total}	
	items.append(record_count)
	return jsonify(items), 200

##############################
# Get Single Data from any Model
##############################
@app.route('/<string:model>/<int:id>/', methods=['GET'])
@jwt_required
@access_check
def get_data(model, id):
	print(get_jwt_identity())
	data = Models[model]['class'].query.get(id)
	#if not data:
	#	return not_found_message(model, id), 404
	return Models[model]['schema'].jsonify(data), 200

##############################
# Update any Model
##############################
@app.route('/<string:model>/<int:id>/', methods=['PUT'])
@jwt_required
@access_check
def update_data(model, id):
	data = Models[model]['class'].query.get(id)
	if not data:
		return not_found_message(model, id), 404

	if 'update' in dir(Models[model]['class']):
		Models[model]['class'].update(data, request.json)
	else:
		for field in request.json:
			setattr(data, field, request.json.get(field))

	db.session.commit()
	return Models[model]['schema'].jsonify(data), 200

##############################
# Delete any Model
##############################
@app.route('/<string:model>/<int:id>/', methods=['DELETE'])
@jwt_required
@access_check
def delete_data(model, id):
	data = Models[model]['class'].query.get(id)
	if not data:
		return not_found_message(model, id), 404

	db.session.delete(data)
	db.session.commit()
	return Models[model]['schema'].jsonify(data), 200


##############################
# Error Pages
##############################

@ app.errorhandler(404)
def err_404(error):
	return render_template('error.html', message='Page not found'), 404

@ app.errorhandler(500)
def err_500(error):
	return render_template('error.html', message='Internal server error'), 500

