/*****************************************************************************
*
* - File              : main.c
*
* - Compiler          : gcc version 3.4.6
*
* - Support mail      : Koryagin Andrey
*                     : andre@avislab.com
*
* - Supported devices : ATmega168
*
* - Description       : ATmega168 for sensored BLDC
*                       control of a three phase brushless DC motor.
*
*****************************************************************************/
#include "BLDC.h"

#include <stdio.h>
#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>

// Flags
unsigned char				FLAGS[5];
#define FLAG_RUN			FLAGS[0]
#define FLAG_RefUpdated		FLAGS[1]
#define FLAG_SpeedUpdated	FLAGS[2]
#define FLAG_CurrentUpdated	FLAGS[3]
#define FLAG_VoltageUpdated	FLAGS[4]

// LED Modes
#define LED_MODE_OFF			0
#define LED_MODE_ON				1
#define LED_MODE_BLINK			2
#define LED_MODE_FAST_BLINK		3
#define LED_MODE_SPLASH			4

// Alert Mode
#define ALERT_NULL				0
#define ALERT_SPEED				1
#define ALERT_CURRENT			2
#define ALERT_VOLTAGE			3

// Motor data
typedef struct {
	unsigned char	Direction;
	unsigned char	Duty;
	unsigned long	Speed;
	unsigned int	Current;
	unsigned int	Voltage;
	unsigned char	PWM_Top_Value;
	unsigned char	PWM_Min_Value;
	unsigned char	PWM_Startup_Value;
	unsigned char	Start_Ref;
	unsigned char	Stop_Ref;
	unsigned char	Alert_Mode;
} motor_var;
motor_var Motor;


// SETTINGS
typedef struct {
	unsigned char	DutyDamper;
	unsigned char	Magnets;
	unsigned char	PWM_Startup; // (%)
	unsigned char	PWM_Min; // (%)
	unsigned char	Voltage_Min;
	unsigned char	Current_Max;
	unsigned char	DeadTime; // 

} motor_settings;
motor_settings Settings;


// Set default settings
void DefaultSettings() {
	Settings.DutyDamper = 250;
	Settings.Magnets = 12;
	Settings.PWM_Startup = 25;
	Settings.PWM_Min = 10;
	Settings.Voltage_Min = 10; // 10 //V=((Voltage_Min*0.0196)/R2) * (R1+R2)   // Voltage_Min = (V/(R1+R2)*R2)/0.0196
	Settings.Current_Max = 180;
	Settings.DeadTime = 2;
}

// Preset motor data
void PrepareSettings() {
	Motor.Direction = CCW;
	Motor.Duty = 0;
	Motor.PWM_Top_Value = PWM_TOP_VALUE_16K;
	// PWM START
	Motor.PWM_Startup_Value = (unsigned char) ((Settings.PWM_Startup*Motor.PWM_Top_Value)/100);
    // PWM STOP
    Motor.PWM_Min_Value = (unsigned char) ((Settings.PWM_Min*Motor.PWM_Top_Value)/100);

	Motor.Start_Ref = POT_START_REF;
	Motor.Stop_Ref = POT_STOP_REF;
}

unsigned char duty_damper_counter;

// Array of power stage enable signals for each commutation step
unsigned char driveTable[6];

// Array of ADC channel selections for each commutation step
unsigned char ADMUXValue;
unsigned char ADMUXIndex=0;

unsigned int timeSinceCommutation;

// ADC reading of external analog speed reference
volatile unsigned char speedReference;

// ADC reading of shunt voltage.
volatile unsigned char currentVoltageADC;

// ADC reading of the known external reference voltage
volatile unsigned char batteryVoltageADC;

unsigned char LED_Mode;
unsigned char LED_Enabled;
unsigned char LED_ON_Time;
unsigned char LED_OFF_Time;
unsigned char LED_Counter;


// Read ADC
unsigned char adc_read(unsigned char ch) {
	ADMUX = (1 << REFS0) | (1 << ADLAR) | (ch & 0x1F);	// set channel (VREF = VCC) , left align, chennel

	// Start first ADC conversion
	ADCSRA = (1 << ADEN) | (1 << ADSC) | (0 << ADATE) | (0 << ADIF) | (0 << ADIE) | (0 << ADPS2) | (1 << ADPS1) | (0 << ADPS0);
	while(ADCSRA & (1 << ADSC));		// wait until conversion complete

	// Result is not correct after first conversation
	// Restart ADC conversion
	ADCSRA = (1 << ADEN) | (1 << ADSC) | (0 << ADATE) | (0 << ADIF) | (0 << ADIE) | (0 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);
	while(ADCSRA & (1 << ADSC));		// wait until conversion complete	

	return ADCH;
}

// Set LED mode
void Set_LED_Mode(unsigned char mode) {
	LED_Mode = mode;

	if (mode==LED_MODE_OFF) {
		LED_OFF;
	}
	
	if (mode==LED_MODE_ON) {
		LED_ON;
	}

	if (mode==LED_MODE_BLINK) {
		LED_ON;
		LED_Enabled = TRUE;
		LED_ON_Time = 200;
		LED_OFF_Time = 200;
	}

	if (mode==LED_MODE_FAST_BLINK) {
		LED_ON;
		LED_Enabled = TRUE;
		LED_ON_Time = 20;
		LED_OFF_Time = 20;
	}

	if (mode==LED_MODE_SPLASH) {
		LED_ON;
		LED_Enabled = TRUE;
		LED_ON_Time = 20;
		LED_OFF_Time = 200;
	}

	LED_Counter = 0;
}


// Main
int main(void)
{
	DefaultSettings();
	PrepareSettings();

	InitPorts();
	InitTimers();
	MakeTables();
  
	PCICR |= (1<<PCIE1);
	PCMSK1 |= (1<<PCINT8) | (1<<PCINT9) | (1<<PCINT10);

	FLAG_RUN = FALSE;
	sei();

	Set_LED_Mode(LED_MODE_ON);
  
	speedReference = 0;
 
	for(;;)
	{
		// Motor
		if (FLAG_RUN == FALSE) {
			currentVoltageADC = adc_read(ADMUX_CURRENT);
			speedReference = adc_read(ADMUX_SPEED_REF);
			batteryVoltageADC = adc_read(ADMUX_BATTERY_VOLTAGE);

			CalculateCurrent();
			CalculateVoltage();
			Motor.Speed = 0;

			if ((Motor.Alert_Mode ==0) & (speedReference > Motor.Start_Ref)) {
				if (BTN_IS_DOWN) {
					Motor.Direction = CW;
				}
				else {
					Motor.Direction = CCW;
				}

				MakeTables();
				StartMotor();
				FLAG_RUN = TRUE;
			}
		}
		else {
			PWMControl();
		}

		// Current Calculate
		if (FLAG_CurrentUpdated == TRUE) {
			//CalculateCurrent();
			FLAG_CurrentUpdated = FALSE;
		}

		// Voltage Calculate
		if (FLAG_VoltageUpdated == TRUE) {
			//CalculateVoltage();
			FLAG_VoltageUpdated = FALSE;
		}

		// Speed Calculate
		if (FLAG_SpeedUpdated == TRUE) {
			//CalculateSpeed();
			FLAG_SpeedUpdated = FALSE;
		}

		// Alerts
		// Speed Limit
		if (Motor.Alert_Mode == ALERT_SPEED) {
			ALARMStopMotor(LED_MODE_SPLASH);
		}

		//Current Limit
		if (Motor.Alert_Mode == ALERT_CURRENT) {
			ALARMStopMotor(LED_MODE_FAST_BLINK);
		}

		// Voltage Limit
		if (Motor.Alert_Mode == ALERT_VOLTAGE) {
			ALARMStopMotor(LED_MODE_BLINK);
		}
	}
}

/*****************************************************************************/


// Initializes I/O ports
static void InitPorts(void)
{
	// Init DRIVE_DDR for motor driving.
	DRIVE_DDR = (1 << UL) | (1 << UH) | (1 << VL) | (1 << VH) | (1 << WL) | (1 << WH);

	// Init PORTD for PWM on PD5.
	DDRD |= (1 << PD5);

  	// Init LED Port
	LED_DDR |= (1<<LED_PIN);
	LED_OFF;

	// Init Button Port
	BTN_PORT |= (1<<BTN_PIN);
}


// Initializes timers
static void InitTimers(void)
{
	// Set up Timer/counter0 for PWM, output on OCR0B, OCR0A as TOP value, prescaler = 1.
	TCCR0A = (0 << COM0A1) | (0 << COM0A0) | (1 << COM0B1) | (0 << COM0B0) | (0 << WGM01) | (1 << WGM00);
	TCCR0B = (1 << WGM02) | (0 << CS02) | (1 << CS01) | (0 << CS00);

	OCR0A = Motor.PWM_Top_Value;
	//TIFR0 = TIFR0;
	TIMSK0 = (0 << TOIE0);

	// Set up Timer/counter1 for commutation timing, prescaler = 8.
	TCCR1B = (1 << CS11) | (0 << CS10);

	// Set up Timer2
	TCCR2B = ((1<<CS22) | (0<<CS21) | (1<<CS20)); // prescaler = 128
	TIMSK2|=(1<<TOIE2);
}



// Initializes arrays for motor driving and AD channel selection
static void MakeTables(void)
{
  if (Motor.Direction == CCW) {
    driveTable[0] = DRIVE_PATTERN_STEP1_CCW;
    driveTable[1] = DRIVE_PATTERN_STEP2_CCW;
    driveTable[2] = DRIVE_PATTERN_STEP3_CCW;
    driveTable[3] = DRIVE_PATTERN_STEP4_CCW;
    driveTable[4] = DRIVE_PATTERN_STEP5_CCW;
    driveTable[5] = DRIVE_PATTERN_STEP6_CCW;
  }
  else {
	driveTable[0] = DRIVE_PATTERN_STEP4_CW;
    driveTable[1] = DRIVE_PATTERN_STEP3_CW;
    driveTable[2] = DRIVE_PATTERN_STEP2_CW;
    driveTable[3] = DRIVE_PATTERN_STEP1_CW;
    driveTable[4] = DRIVE_PATTERN_STEP6_CW;
    driveTable[5] = DRIVE_PATTERN_STEP5_CW;
  }
}

// Set port value, using deadtime
static void SET_DRIVE_PORT(unsigned char DrivePattern) {
  char i;

  DRIVE_PORT = (DRIVE_PORT && DrivePattern);

  for (i=0; i<Settings.DeadTime; i++) {
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");

		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
		asm("nop");
	}

	DRIVE_PORT = DrivePattern;

}

// Set port value by Hall sensors data
void SENSORS_ISR() {
	unsigned char position;
	unsigned char DrivePattern;
	position = (PINC&0x07);
	
	switch(position)
	{
		case 5: DrivePattern = driveTable[0]; break;
		case 1: DrivePattern = driveTable[1]; break;
		case 3: DrivePattern = driveTable[2]; break;
		case 2: DrivePattern = driveTable[3]; break;
		case 6: DrivePattern = driveTable[4]; break;
		case 4: DrivePattern = driveTable[5]; break;
		default: DrivePattern = 0; break;
	}
	
	SET_DRIVE_PORT(DrivePattern);
}


// Executes the motor startup sequence
static void StartMotor(void)
{
	Motor.Duty = Motor.PWM_Startup_Value;
	SET_PWM_COMPARE_VALUE(Motor.Duty);
	timeSinceCommutation = 0;
	Motor.Alert_Mode = ALERT_NULL;
	ENABLE_TIMER0_INT;
	SENSORS_ISR();
}

// Stop motor
static void StopMotor(void)
{
	FLAG_RUN = FALSE;
	DISABLE_DRIVING;
	DISABLE_ALL_TIMER0_INTS;
	Motor.Duty = 0;
	SET_PWM_COMPARE_VALUE(0);
	timeSinceCommutation = 0;
}

// Stop motor and enable LED blink mode
static void ALARMStopMotor(unsigned char mode)
{
	StopMotor();
	Set_LED_Mode(mode);
	while (adc_read(ADMUX_SPEED_REF) > Motor.Start_Ref) {

	}
	Delay(100); // Delay after set REF to zero.
	Motor.Alert_Mode = 0;
	Set_LED_Mode(LED_MODE_ON);
}

// Timer0 bottom overflow
ISR(TIMER0_OVF_vect)
{
	if (ADMUXIndex == 0) {


		currentVoltageADC = adc_read(ADMUX_CURRENT);

		if (currentVoltageADC > Settings.Current_Max) {
			StopMotor();
			Motor.Alert_Mode = ALERT_CURRENT;
			return;
		}

		FLAG_CurrentUpdated = TRUE;
	}

	if (ADMUXIndex == 1) {
		speedReference = adc_read(ADMUX_SPEED_REF);
		FLAG_RefUpdated = TRUE;
	}
  
	if (ADMUXIndex == 2) {
		batteryVoltageADC = adc_read(ADMUX_BATTERY_VOLTAGE);
		if (batteryVoltageADC < Settings.Voltage_Min) {
			StopMotor();
			Motor.Alert_Mode = ALERT_VOLTAGE;
			return;
		}
		FLAG_VoltageUpdated = TRUE;
	}

	ADMUXIndex++;
	if (ADMUXIndex > 2) {
		ADMUXIndex = 0;
	}


}

// Timer2 overflow
ISR(TIMER2_OVF_vect) {
	if (LED_Mode > 1) {
		LED_Counter++;
		if (LED_Enabled==TRUE) {
			if (LED_Counter > LED_ON_Time) {
				LED_OFF;
				LED_Enabled=FALSE;
				LED_Counter = 0;
			}
		}
		else {
			if (LED_Counter > LED_OFF_Time) {
				LED_ON;
				LED_Enabled=TRUE;
				LED_Counter = 0;
			}
		}
	}
}

// Interrupt from Hall sensors
ISR(PCINT1_vect) {
	unsigned int timeSinceCommutation;
	SENSORS_ISR();

	// Find time since last commutation
	timeSinceCommutation = TCNT1;
	TCNT1 = 0;

	FLAG_SpeedUpdated = TRUE;
}


// Generates a delay 
void Delay(unsigned int delay)
{
	do
	{
		TCNT1 = 0xffff - 200;
		// Wait for timer to overflow.
		while (!(TIFR1 & (1 << TOV1)))
		{

		}

		delay--;
	} while (delay);
}


// Update duty
static void PWMControl(void)
{
	unsigned char duty;
  
	// Only update duty cycle if a new speed reference measurement has been made. (Done right after speed measurement is ready)
	if (FLAG_RefUpdated)
	{
		if (speedReference < Motor.Stop_Ref) {

				SET_PWM_COMPARE_VALUE(0);
				Motor.Duty = 0;
		}
		else {
			duty = (unsigned char) (Motor.PWM_Min_Value + speedReference * (Motor.PWM_Top_Value - Motor.PWM_Min_Value) / ADC_RESOLUTION);
			if (duty_damper_counter > Settings.DutyDamper) {
  				if (duty > Motor.Duty) {
  					Motor.Duty++;
  				}
  				else {
					Motor.Duty--;
  				}
   				duty_damper_counter = 0;
			}
			duty_damper_counter++;

			SET_PWM_COMPARE_VALUE(Motor.Duty);
		}
	}
	FLAG_RefUpdated = FALSE;
}


// Calculates the current speed in electrical RPM
void CalculateSpeed()
{
/*
	unsigned long rotationPeriod;

	rotationPeriod = (unsigned long)timeSinceCommutation * 6 * Settings.Magnets;
	Motor.Speed = (TICKS_PER_MINUTE / rotationPeriod);
*/
}

// Calculates current consumption
void CalculateCurrent()
{
	//Motor.Current = abs((int)(currentVoltageADC) - 127) * 1;
}

// Calculate battery voltage
void CalculateVoltage()
{
	//Motor.Voltage = (unsigned int)((EXTERNAL_REF_VOLTAGE/10 * (unsigned long)batteryVoltageADC * (RESISTOR_R1 + RESISTOR_R2)) / ADC_RESOLUTION) / RESISTOR_R2;
}
