#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "onewire.h"
#include "ds18x20.h"
#include "delay.h"
#include "lcd_lib.h"
#include "uart.h"

#define BAUD 9200
#define UBRR (F_CPU/16/BAUD)-1
#define RX_BUF_SIZE 20

char bufferRX[RX_BUF_SIZE];
uint8_t iRX;
char cRX;

#define MAXSENSORS 1

const uint8_t mmrtst[] PROGMEM ="\0";

const uint8_t gradus[] PROGMEM= {
0b00001000,
0b00010100,
0b00001000,
0b00000000,
0b00000000,
0b00000000,
0b00000000,
0b00000000
};

const uint8_t battery0[] PROGMEM= {
0b00001110,
0b00011011,
0b00010001,
0b00010001,
0b00010001,
0b00010001,
0b00010001,
0b00011111
};

const uint8_t battery25[] PROGMEM= {
0b00001110,
0b00011011,
0b00010001,
0b00010001,
0b00010001,
0b00010001,
0b00011111,
0b00011111
};

const uint8_t battery50[] PROGMEM= {
0b00001110,
0b00011011,
0b00010001,
0b00010001,
0b00011111,
0b00011111,
0b00011111,
0b00011111
};

const uint8_t battery75[] PROGMEM= {
0b00001110,
0b00011011,
0b00010001,
0b00011111,
0b00011111,
0b00011111,
0b00011111,
0b00011111
};

const uint8_t battery100[] PROGMEM= {
0b00001110,
0b00011111,
0b00011111,
0b00011111,
0b00011111,
0b00011111,
0b00011111,
0b00011111
};

uint8_t gSensorIDs[MAXSENSORS][OW_ROMCODE_SIZE];
uint8_t buffer[8];

#define AGOV_BUFFER_SIZE 24
uint8_t AGOV_FLAG;
uint8_t OK_FLAG;
uint8_t buffer_agov[AGOV_BUFFER_SIZE];
uint8_t iAGOV;


uint8_t time_counter;
uint32_t time_sec;

///////////////////////////////////////////
//         0
///////////////////////////////////////////
void clear_agov_buffer(void) {
  for (iAGOV=0;iAGOV<AGOV_BUFFER_SIZE;iAGOV++)
    buffer_agov[iAGOV] = 0;
  iAGOV = 0;
}

//    
void str_to_agov_buffer(uint8_t *_buffer) {
  uint8_t i;
  i=0;
  while ((_buffer[i]!=0) && (iAGOV < AGOV_BUFFER_SIZE)) {
    if (_buffer[i] != 32) {
      buffer_agov[iAGOV] = _buffer[i];
	  iAGOV++;
	}
	i++;
  }
}

///////////////////////////////////////////
//    
///////////////////////////////////////////
void char_to_agov_buffer(char c) {
  if (iAGOV < AGOV_BUFFER_SIZE)
  {
    buffer_agov[iAGOV] = c;
    iAGOV++;
  }
}

///////////////////////////////////////////
//         LCD
// _buffer - ,    
// _value -  ( 65536)
// _n -  .     ,    . ( 5)
///////////////////////////////////////////
void intToStr(uint8_t* _buffer, uint16_t _value, uint8_t _n) {
  uint8_t i,j;
  uint16_t num;
  
  for (i=0; i<_n; i++) {
    //        ( _value<0 )
	if (/*(i==0) &&*/ (_value != abs(_value))) {
	  _value = abs(_value);
	  _buffer[i] = 45;
	} else {
	  num = 1;
	  for (j=1; j< (_n-i); j++) {
	    num = num*10;
	  }
	  _buffer[i] = (_value / num);
	  _value -= _buffer[i] * num;
	  _buffer[i] += 48;
	}
  }
}

///////////////////////////////////////////
//       
///////////////////////////////////////////
void ZeroToSpace(uint8_t* _buffer, uint8_t _n) {
  uint8_t i;
  i = 0;
  while ((i<_n) & ( (_buffer[i]==48) || (_buffer[i]==45) )) {
    if (_buffer[i]==48) {
	  _buffer[i] = 32;
	}
    i++;
  }
  //       
  if (_buffer[0]==45) {
    _buffer[0] = 32;
	_buffer[i-1] = 45;
  }
}


void floatToStr(uint8_t* _buffer, float _value, uint8_t _n, uint8_t _nPoint) {
  uint8_t i;

  //   10   .   
  if (_nPoint > 0) {
    for (i=0; i<_nPoint+1; i++) {
      _value *= 10;
    }
  }

  intToStr(_buffer, _value, _n);

  if (_nPoint > 0) {
    ZeroToSpace(_buffer, _n-_nPoint-2);
  }
  else {
    ZeroToSpace(_buffer, _n-1);
  }


  if (_nPoint > 0) {
    for (i=1; i<=_nPoint; i++) {
      _buffer[_n-i] = _buffer[_n-i-1];
	  _buffer[_n] = 0;
    }
	_buffer[_n-i] = '.';
  }
}


///////////////////////////////////////////
//      
///////////////////////////////////////////
void SecondsToTime (uint8_t* _buffer, uint32_t _Seconds) {
  uint32_t h,m,s;  
  
  h = _Seconds / 3600;
  m = (_Seconds - h*3600) / 60;
  s = _Seconds - h*3600 - m*60;

  intToStr(_buffer, s, 2);
  _buffer[7] = _buffer[1];
  _buffer[6] = _buffer[0];
  _buffer[5] = ':';

  intToStr(_buffer, m, 2);
  _buffer[4] = _buffer[1];
  _buffer[3] = _buffer[0];
  _buffer[2] = ':';
   
   intToStr(_buffer, h, 2);
}

//    
uint32_t pow_10(uint8_t i)
{
  uint16_t uiData;

  if (i==0) return 1;

  uiData=10;
  i=i-1;
  while(i) {
	uiData *= 10; 
    i--;
  }
  return uiData;
}



uint32_t StrToInt (uint8_t* _buffer) {
  uint32_t result;
  uint8_t x10;
  uint8_t pos;
  
  result = 0;
  pos = 0;

  //  
  while (_buffer[pos] != 0) {
    pos++;
  }

  x10 = 0;
  pos--;
  
  while ((_buffer[pos] > 47) && (_buffer[pos] < 58) && (pos >= 0)) {
	result += (_buffer[pos]-48) * pow_10(x10);
	pos--;
	x10++;
  }

  return result;
}

//===================================  ==================================
// :
void adc_init(void){	

  ADCSRA = _BV(ADEN) | _BV(ADPS0) | _BV(ADPS1) | _BV(ADPS2); // prescaler = 128

}


//  :

uint32_t adc_read(uint8_t ch){
  ADMUX = _BV(REFS0) | (ch & 0x1F);	// set channel (VREF = VCC)	
  ADCSRA &= ~_BV(ADIF);			// clear hardware "conversion complete" flag	
  ADCSRA |= _BV(ADSC);			// start conversion	

  while(ADCSRA & _BV(ADSC));		// wait until conversion complete	

  return ADC;				// read ADC (full 10 bits);
}


//  :

uint32_t adc_mv(uint8_t ch){	
  return 5000*adc_read(ch) / 0x3FF;
}

//=================================== DS18B20 ==================================

uint8_t search_sensors(void)
{
	uint8_t i;
	uint8_t id[OW_ROMCODE_SIZE];
	uint8_t diff, nSensors;
	
	nSensors = 0;
	
	for( diff = OW_SEARCH_FIRST; 
		diff != OW_LAST_DEVICE && nSensors < MAXSENSORS ; )
	{
		DS18X20_find_sensor( &diff, &id[0] );
		
		if( diff == OW_PRESENCE_ERR ) {
			break;
		}
		
		if( diff == OW_DATA_ERR ) {
			break;
		}
		
		for (i=0;i<OW_ROMCODE_SIZE;i++)
			gSensorIDs[nSensors][i]=id[i];
		
		nSensors++;
	}	
	return nSensors;
}

///////////////////////////////////////////
//         0
///////////////////////////////////////////
void clear_bufferRX(void) {
  for (iRX=0;iRX<RX_BUF_SIZE;iRX++)
    bufferRX[iRX] = 0;
  iRX = 0;
}

///////////////////////////////////////////
//    
///////////////////////////////////////////
ISR(TIMER1_OVF_vect) { //  100   
  uint8_t str[8];
  
  TCNT1 = 0x10000 - (F_CPU/1024/100);
  cli();

  cRX = uart_getc();
  if ((cRX != 255) && (cRX != 0)) {
    uart_putc(cRX);
	
	if ((cRX != 13) && (cRX != 10)) {  //   
      bufferRX[iRX] = cRX;
      iRX++;
    }
    else {	  
	  if ((strcmp(bufferRX,"AGOV") == 0) || (strcmp(bufferRX,"agov") == 0)) { // AGOV
		AGOV_FLAG = 1;
	  }

	  
      strncpy (str,bufferRX,7);
      str[7]='\0';	  
	  if ((strcmp(str,"SETTIME") == 0) || (strcmp(str,"settime") == 0)) { // settime
        time_sec = StrToInt(bufferRX);
		OK_FLAG = 1;
	  }
	  clear_bufferRX();
    }	
  }
  if (iRX > RX_BUF_SIZE-1) {
    clear_bufferRX();
  }


  time_counter++;
  if (time_counter > 100) {
    time_counter = 0;
	time_sec++;
	if (time_sec > 86400) {
      time_sec = 0;
	}
  }
  sei();
}

//=================================== MAIN ==================================
int main( void )
{
char cRX;	
	float CurrentTemp;
	uint32_t adc_value;
	uint8_t nSensors, i;
	uint8_t subzero, cel, cel_frac_bits;
    	
	LCDinit();    
	LCDdefinechar(gradus,0x0F);
	LCDdefinechar(battery0,0x00);
	LCDdefinechar(battery25,0x01);
	LCDdefinechar(battery50,0x02);
	LCDdefinechar(battery75,0x03);
	LCDdefinechar(battery100,0x4);
	LCDcursorOFF();
    LCDclr();

	sei();	
	
    adc_init();

	nSensors = search_sensors(); 
	
    uart_init (UBRR); //  uart	
    AGOV_FLAG = 0;
    OK_FLAG = 0;

    //  
    TIMSK = (1<<TOIE1);
    TCCR1B = ((1<<CS10) | (1<<CS12));
    //  
    TCNT1 = 0x10000 - (F_CPU/1024/100); 

	while(1) {				// main loop		
		clear_agov_buffer();
		for ( i=nSensors; i>0; i-- ) {
			if ( DS18X20_start_meas( DS18X20_POWER_PARASITE, &gSensorIDs[i-1][0] ) == DS18X20_OK ) {
				delay_ms(DS18B20_TCONV_12BIT);
				if ( DS18X20_read_meas( &gSensorIDs[i-1][0], &subzero, &cel, &cel_frac_bits) == DS18X20_OK ) {
           			CurrentTemp = cel_frac_bits;
					CurrentTemp = cel + CurrentTemp/10;

					if (subzero == 1)
						CurrentTemp = -1*CurrentTemp;

					floatToStr(buffer, CurrentTemp, 5 ,1);
					LCDGotoXY((16-8*i) + 1,0);
					LCDstring(buffer,5);
					LCDsendChar(0x0F);
		
                    str_to_agov_buffer(buffer);
                    char_to_agov_buffer(';');
				}
			}
		}		
				
		//  
		
		//  %
		// PH=33,5(Vout - 0,773)
		adc_value = adc_mv(0);
		adc_value = (335*(adc_value - 773))/10000;
		if (adc_value > 100) adc_value = 100;
        LCDGotoXY(0,1);
		intToStr(buffer, adc_value, 3);
		ZeroToSpace(buffer,2);
		LCDstring(buffer,3);
		buffer[3] = 0;
        str_to_agov_buffer(buffer);
        char_to_agov_buffer(';');
		LCDsendChar('%');

		// 
		//adc_value = adc_read(1);
		//value = (111*(((adc_value*100000)/1024)+95))/100; //   
		adc_value = adc_mv(2);
		adc_value = 750064*(adc_value/5+95)/900000; //   ... () K=7.50064

        LCDGotoXY(10,1);
		intToStr(buffer, adc_value, 3);
		ZeroToSpace(buffer,2);
		LCDstring(buffer,3);
		buffer[3] = 0;
        str_to_agov_buffer(buffer);
        CopyStringtoLCD(mmrtst,14,1);
				
		// 
		adc_value = adc_read(1);
        adc_value = ((adc_value -512)*100)/410; //  %. 0% -   
		LCDGotoXY(15,0);
		cel = 0x00;
		for (i=0; i<4; i++) {
		  if (adc_value > i*25) cel = i;
		}
		if (adc_value > 85) cel = 0x04;
		LCDsendChar(cel);


		//		
		SecondsToTime (buffer, time_sec);
		LCDGotoXY(0,0);
		LCDstring(buffer,8);

		//delay_ms(1000);
       // ------------------------------------
		if (AGOV_FLAG == 1) {
		  AGOV_FLAG = 0;
		  uart_puts("***");
		  uart_puts(buffer_agov);
		  uart_puts("***\r\n");
		}

		if (OK_FLAG == 1) {
		  OK_FLAG = 0;

SecondsToTime (buffer, time_sec);
//intToStr(buffer, time_sec, 5);
//buffer[5] = "\0";
uart_puts(buffer);

		  uart_puts("\r\nOK\r\n");
		}

	}
	
}
